from codeable_detectors.basic_detectors import AtLeastOneFileMatchesDetector
from pyparsing import Literal, Optional, oneOf

from codeable_detectors.evidences import ServiceEvidence, FailedEvidence
from codeable_detectors.java.java_detectors import detect_java_import
from codeable_detectors.pyparsing_patterns import round_braces_block


class JavaSparkService(AtLeastOneFileMatchesDetector):
    def __init__(self):
        super().__init__()
        self.file_endings = ["java"]

    def detect_in_context(self, ctx, **kwargs):
        package_name = "spark"
        class_name = "Spark"
        fully_qualified_name = package_name + "." + class_name

        import_matches = detect_java_import(ctx, package_name, class_name)
        import_detected = False
        static_import_detected = False

        if import_matches:
            import_detected = True
            for importMatch in import_matches:
                if importMatch.kwargs["isStaticImport"]:
                    static_import_detected = True

                    # if no import evidence is found, still match e.g.: spark.Spark.get(...);
        class_name_pattern = Literal(fully_qualified_name) + Literal(".")
        if import_detected:
            # match e.g.: spark.Spark.get(...) and Spark.get(...);
            class_name_pattern = (Literal(fully_qualified_name) | Literal(class_name)) + Literal(".")
            if static_import_detected:
                # match example, additionally: get(...);
                class_name_pattern = Optional(class_name_pattern)

        spark_http_method_matches = ctx.matches_pattern(class_name_pattern +
                                                        oneOf(
                                                            "get put post delete head") + round_braces_block +
                                                        Literal(";"))
        if spark_http_method_matches:
            matches = []
            if import_detected:
                matches = import_matches
            matches.extend(spark_http_method_matches)
            return ServiceEvidence(matches).set_properties(
                detector_technology_types=["java", "spark", "restful"], kwargs=kwargs)
        return FailedEvidence("no call to Java Spark HTTP methods found")
